"""
Utilities for constructing a non‑empty S⁺ mask when the standard
compact‑curvature translator fails to find any signal.  This helper
implements a simple heuristic fallback that operates directly on the
raw E0 snapshot.  It attempts to infer a meaningful region by
examining multi‑scale Laplacian‑of‑Gaussian (LoG) responses and, if
those responses are identically zero, falls back to selecting a
compact central region using a radial weighting.  The overall goal is
to produce a non‑empty mask without introducing any synthetic E0
content or relying on precomputed kernels.

This module is only invoked when ``build_mask`` returns an empty mask.
The caller should still honour any quality control checks and bail if
this fallback also fails to produce a non‑empty region.
"""

from __future__ import annotations

import numpy as np
from typing import Tuple

try:
    from scipy.ndimage import gaussian_laplace, binary_opening, binary_closing, binary_fill_holes, label
except Exception:
    # The driver requires SciPy; if unavailable the fallback cannot
    # operate and will return an empty mask.
    gaussian_laplace = None  # type: ignore
    binary_opening = None  # type: ignore
    binary_closing = None  # type: ignore
    binary_fill_holes = None  # type: ignore
    label = None  # type: ignore

def compute_fallback_mask(E0: np.ndarray,
                          sigma_list: Tuple[float, ...] | None = None,
                          pos_tail_quantile: float = 0.99,
                          fallback_frac: float = 0.005) -> np.ndarray:
    """
    Build a fallback S⁺ mask from an E0 snapshot.  The algorithm
    proceeds as follows:

    * Standardize ``E0`` via z‑score.  If the array has zero variance
      the standardized result will be identically zero.
    * Compute multi‑scale Laplacian‑of‑Gaussian responses for the
      specified ``sigma_list`` (defaults to (1,2,3)).  The absolute
      value of each response is taken and the maximum across scales
      forms a scale‑invariant response ``S``.
    * Threshold the positive tail of ``S`` at the given quantile.  If
      that yields a non‑empty mask, use it.  Otherwise proceed to a
      final fallback.
    * Final fallback: select the top ``fallback_frac`` fraction of
      pixels according to a radial weighting.  When ``S`` is constant
      (e.g. ``E0`` is constant) all values are identical, so we add a
      small radial weight favouring the centre of the array.  This
      ensures that the selected region is compact and central rather
      than randomly scattered.
    * Apply a light opening and closing to remove single pixel noise
      and fill holes, then keep only the largest connected component.

    Parameters
    ----------
    E0 : np.ndarray
        Two‑dimensional snapshot array of shape (L,L).
    sigma_list : tuple of float, optional
        Standard deviations at which to compute the LoG.  Defaults to
        (1, 2, 3).  Larger values broaden the response and may
        increase coverage.
    pos_tail_quantile : float, optional
        Quantile of the positive tail used for initial thresholding.
        Should be in (0,1).  Defaults to 0.99, corresponding to the
        99th percentile of positive values.
    fallback_frac : float, optional
        Fraction of pixels to retain in the final fallback.  Defaults
        to 0.005 (0.5%).

    Returns
    -------
    np.ndarray
        Binary mask of the same shape as ``E0``.  If the fallback
        fails (e.g. SciPy is unavailable) the returned mask will
        contain all zeros.
    """
    if gaussian_laplace is None:
        # Without SciPy we cannot compute LoG.  Return empty mask.
        return np.zeros_like(E0, dtype=np.uint8)

    # Ensure input is float64 for numerical stability
    X = np.asarray(E0, dtype=np.float64)
    L = int(X.shape[0])
    # Z‑score normalization; avoid division by zero
    m = float(np.mean(X))
    sd = float(np.std(X))
    Xz = (X - m) / (sd if sd > 0 else 1.0)

    # Multi‑scale absolute LoG response
    sigmas = sigma_list or (1.0, 2.0, 3.0)
    responses = []
    for s in sigmas:
        try:
            resp = gaussian_laplace(Xz, sigma=float(s))
        except Exception:
            resp = np.zeros_like(Xz)
        responses.append(np.abs(resp))
    S = np.max(np.stack(responses, axis=0), axis=0)

    # Initial threshold on positive tail
    pos = S[S > 0]
    if pos.size > 0:
        try:
            thr = np.quantile(pos, pos_tail_quantile)
        except Exception:
            thr = None
    else:
        thr = None
    if thr is not None and np.isfinite(thr) and thr > 0:
        mask = (S >= thr)
    else:
        mask = np.zeros_like(S, dtype=bool)

    # Fallback to radial weighting if mask is empty
    if not mask.any():
        # Compute radial distance from centre
        yy, xx = np.ogrid[:L, :L]
        cy = (L - 1) / 2.0
        cx = (L - 1) / 2.0
        r = np.sqrt((yy - cy) ** 2 + (xx - cx) ** 2)
        # Normalise distances so that centre has largest weight
        r_norm = r.max() - r
        # Combine the response with radial weight (tiny epsilon to
        # preserve ordering among identical values)
        weight = S + 1e-9 * r_norm
        k = max(1, int(float(fallback_frac) * weight.size))
        # Select top k pixels by weight
        idx = np.argpartition(weight.ravel(), -k)[-k:]
        mask = np.zeros_like(S, dtype=bool)
        mask.ravel()[idx] = True

    # Morphological clean‑up: opening then closing
    if binary_opening is not None and binary_closing is not None:
        mask = binary_opening(mask)
        mask = binary_closing(mask)
    if binary_fill_holes is not None:
        mask = binary_fill_holes(mask)

    # Keep the largest connected component
    if label is not None:
        lbl, num = label(mask)
        if num > 1:
            sizes = np.bincount(lbl.ravel())
            sizes[0] = 0  # background
            keep = np.argmax(sizes)
            mask = (lbl == keep)

    return mask.astype(np.uint8)

__all__ = ["compute_fallback_mask"]